---
title: Upstash · Cloudflare Workers docs
description: Upstash is a serverless database with Redis* and Kafka API. Upstash
  also offers QStash, a task queue/scheduler designed for the serverless.
lastUpdated: 2025-06-11T17:40:43.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/workers/databases/third-party-integrations/upstash/
  md: https://developers.cloudflare.com/workers/databases/third-party-integrations/upstash/index.md
---

[Upstash](https://upstash.com/) is a serverless database with Redis\* and Kafka API. Upstash also offers QStash, a task queue/scheduler designed for the serverless.

## Upstash for Redis

To set up an integration with Upstash:

1. You need an existing Upstash database to connect to. [Create an Upstash database](https://docs.upstash.com/redis#create-a-database) or [load data from an existing database to Upstash](https://docs.upstash.com/redis/howto/connectclient).

2. Insert some data to your Upstash database. You can add data to your Upstash database in two ways:

   * Use the CLI directly from your Upstash console.
   * Alternatively, install [redis-cli](https://redis.io/docs/getting-started/installation/) locally and run the following commands.

   ```sh
   set GB "Ey up?"
   ```

   ```sh
   OK
   ```

   ```sh
   set US "Yo, what’s up?"
   ```

   ```sh
   OK
   ```

   ```sh
   set NL "Hoi, hoe gaat het?"
   ```

   ```sh
   OK
   ```

3. Configure the Upstash Redis credentials in your Worker:

   You need to add your Upstash Redis database URL and token as secrets to your Worker. Get these from your [Upstash Console](https://console.upstash.com) under your database details, then add them as secrets using Wrangler:

   ```sh
   # Add the Upstash Redis URL as a secret
   npx wrangler secret put UPSTASH_REDIS_REST_URL
   # When prompted, paste your Upstash Redis REST URL


   # Add the Upstash Redis token as a secret
   npx wrangler secret put UPSTASH_REDIS_REST_TOKEN
   # When prompted, paste your Upstash Redis REST token
   ```

4. In your Worker, install the `@upstash/redis`, a HTTP client to connect to your database and start manipulating data:

   * npm

     ```sh
     npm i @upstash/redis
     ```

   * yarn

     ```sh
     yarn add @upstash/redis
     ```

   * pnpm

     ```sh
     pnpm add @upstash/redis
     ```

5. The following example shows how to make a query to your Upstash database in a Worker. The credentials needed to connect to Upstash have been added as secrets to your Worker.

   ```js
   import { Redis } from "@upstash/redis/cloudflare";


   export default {
     async fetch(request, env) {
       const redis = Redis.fromEnv(env);


       const country = request.headers.get("cf-ipcountry");
       if (country) {
         const greeting = await redis.get(country);
         if (greeting) {
           return new Response(greeting);
         }
       }


       return new Response("Hello What's up!");
     },
   };
   ```

   Note

   `Redis.fromEnv(env)` automatically picks up the default `url` and `token` names created in the integration.

   If you have renamed the secrets, you must declare them explicitly like in the [Upstash basic example](https://docs.upstash.com/redis/sdks/redis-ts/getstarted#basic-usage).

To learn more about Upstash, refer to the [Upstash documentation](https://docs.upstash.com/redis).

## Upstash Kafka

To set up an integration with Upstash Kafka:

1. Create a [Kafka cluster and topic](https://docs.upstash.com/kafka).

2. Configure the Upstash Kafka credentials in your Worker:

   You need to add your Upstash Kafka connection details as secrets to your Worker. Get these from your [Upstash Console](https://console.upstash.com) under your Kafka cluster details, then add them as secrets using Wrangler:

   ```sh
   # Add the Upstash Kafka URL as a secret
   npx wrangler secret put UPSTASH_KAFKA_REST_URL
   # When prompted, paste your Upstash Kafka REST URL


   # Add the Upstash Kafka username as a secret
   npx wrangler secret put UPSTASH_KAFKA_REST_USERNAME
   # When prompted, paste your Upstash Kafka username


   # Add the Upstash Kafka password as a secret
   npx wrangler secret put UPSTASH_KAFKA_REST_PASSWORD
   # When prompted, paste your Upstash Kafka password
   ```

3. In your Worker, install `@upstash/kafka`, a HTTP/REST based Kafka client:

   * npm

     ```sh
     npm i @upstash/kafka
     ```

   * yarn

     ```sh
     yarn add @upstash/kafka
     ```

   * pnpm

     ```sh
     pnpm add @upstash/kafka
     ```

4. Use the [upstash-kafka](https://github.com/upstash/upstash-kafka/blob/main/README.md) JavaScript SDK to send data to Kafka.

Refer to [Upstash documentation on Kafka setup with Workers](https://docs.upstash.com/kafka/real-time-analytics/realtime_analytics_serverless_kafka_setup#option-1-cloudflare-workers) for more information. Replace `url`, `username` and `password` with the variables set by the integration.

## Upstash QStash

To set up an integration with Upstash QStash:

1. Configure the [publicly available HTTP endpoint](https://docs.upstash.com/qstash#1-public-api) that you want to send your messages to.

2. Configure the Upstash QStash credentials in your Worker:

   You need to add your Upstash QStash token as a secret to your Worker. Get your token from your [Upstash Console](https://console.upstash.com) under QStash settings, then add it as a secret using Wrangler:

   ```sh
   # Add the QStash token as a secret
   npx wrangler secret put QSTASH_TOKEN
   # When prompted, paste your QStash token
   ```

3. In your Worker, install the `@upstash/qstash`, a HTTP client to connect to your database QStash endpoint:

   * npm

     ```sh
     npm i @upstash/qstash
     ```

   * yarn

     ```sh
     yarn add @upstash/qstash
     ```

   * pnpm

     ```sh
     pnpm add @upstash/qstash
     ```

4. Refer to the [Upstash documentation on how to receive webhooks from QStash in your Cloudflare Worker](https://docs.upstash.com/qstash/quickstarts/cloudflare-workers#3-use-qstash-in-your-handler).

\* Redis is a trademark of Redis Ltd. Any rights therein are reserved to Redis Ltd. Any use by Upstash is for referential purposes only and does not indicate any sponsorship, endorsement or affiliation between Redis and Upstash.
